/*****************************************************************
|
|    Copyright 2004-2006 Axiomatic Systems LLC
|
|    This file is part of Melo (Melo AAC Decoder).
|
|    Unless you have obtained Melo under a difference license,
|    this version of Melo is Melo|GPL.
|    Melo|GPL is free software; you can redistribute it and/or modify
|    it under the terms of the GNU General Public License as published by
|    the Free Software Foundation; either version 2, or (at your option)
|    any later version.
|
|    Melo|GPL is distributed in the hope that it will be useful,
|    but WITHOUT ANY WARRANTY; without even the implied warranty of
|    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|    GNU General Public License for more details.
|
|    You should have received a copy of the GNU General Public License
|    along with Melo|GPL; see the file COPYING.  If not, write to the
|    Free Software Foundation, 59 Temple Place - Suite 330, Boston, MA
|    02111-1307, USA.
|
 ****************************************************************/

/*----------------------------------------------------------------------
|       Includes
+---------------------------------------------------------------------*/
#include "MloBitStream.h"
#include "MloConfig.h"
#include "MloDebug.h"
#include "MloHcbQuad.h"
#include "MloDefs.h"

/*----------------------------------------------------------------------
|       Types
+---------------------------------------------------------------------*/
typedef struct MLO_HcbQuad_Binary
{
   MLO_Int8       leaf_flag;
   MLO_Int8       data [4];
}  MLO_HcbQuad_Binary;


typedef struct MLO_HcbQuad_Step1
{
   MLO_Int8       offset;
   MLO_Int8       extra_bits;
}  MLO_HcbQuad_Step1;


typedef struct MLO_HcbQuad_Step2
{
   MLO_Int8       bits;
   MLO_Int8       x;
   MLO_Int8       y;
   MLO_Int8       z;
   MLO_Int8       w;
}  MLO_HcbQuad_Step2;



typedef struct MLO_HcbQuad_Index
{
   const MLO_HcbQuad_Binary *
                  bin_ptr;
   const MLO_HcbQuad_Step1 *
                  step_1_ptr;
   const MLO_HcbQuad_Step2 *
                  step_2_ptr;
}  MLO_HcbQuad_Index;



/*----------------------------------------------------------------------
|       Constants
+---------------------------------------------------------------------*/



/* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -
   Codebooks using binary search tree
*/



/* 4.A, Table 4.A.4 */
static const MLO_HcbQuad_Binary  MLO_HcbQuad_coodbook_3 [] =
{
   { 0, {  1,  2, 0, 0 } },
   { 1, {  0,  0, 0, 0 } },
   { 0, {  1,  2, 0, 0 } },
   { 0, {  2,  3, 0, 0 } },
   { 0, {  3,  4, 0, 0 } },
   { 0, {  4,  5, 0, 0 } },
   { 0, {  5,  6, 0, 0 } },
   { 0, {  6,  7, 0, 0 } },
   { 0, {  7,  8, 0, 0 } },
   { 1, {  1,  0, 0, 0 } },
   { 1, {  0,  0, 0, 1 } },
   { 1, {  0,  1, 0, 0 } },
   { 1, {  0,  0, 1, 0 } },
   { 0, {  4,  5, 0, 0 } },
   { 0, {  5,  6, 0, 0 } },
   { 0, {  6,  7, 0, 0 } },
   { 0, {  7,  8, 0, 0 } },
   { 1, {  1,  1, 0, 0 } },
   { 1, {  0,  0, 1, 1 } },
   { 0, {  6,  7, 0, 0 } },
   { 0, {  7,  8, 0, 0 } },
   { 0, {  8,  9, 0, 0 } },
   { 0, {  9, 10, 0, 0 } },
   { 0, { 10, 11, 0, 0 } },
   { 0, { 11, 12, 0, 0 } },
   { 1, {  0,  1, 1, 0 } },
   { 1, {  0,  1, 0, 1 } },
   { 1, {  1,  0, 1, 0 } },
   { 1, {  0,  1, 1, 1 } },
   { 1, {  1,  0, 0, 1 } },
   { 1, {  1,  1, 1, 0 } },
   { 0, {  6,  7, 0, 0 } },
   { 0, {  7,  8, 0, 0 } },
   { 0, {  8,  9, 0, 0 } },
   { 0, {  9, 10, 0, 0 } },
   { 0, { 10, 11, 0, 0 } },
   { 0, { 11, 12, 0, 0 } },
   { 1, {  1,  1, 1, 1 } },
   { 1, {  1,  0, 1, 1 } },
   { 1, {  1,  1, 0, 1 } },
   { 0, {  9, 10, 0, 0 } },
   { 0, { 10, 11, 0, 0 } },
   { 0, { 11, 12, 0, 0 } },
   { 0, { 12, 13, 0, 0 } },
   { 0, { 13, 14, 0, 0 } },
   { 0, { 14, 15, 0, 0 } },
   { 0, { 15, 16, 0, 0 } },
   { 0, { 16, 17, 0, 0 } },
   { 0, { 17, 18, 0, 0 } },
   { 1, {  2,  0, 0, 0 } },
   { 1, {  0,  0, 0, 2 } },
   { 1, {  0,  0, 1, 2 } },
   { 1, {  2,  1, 0, 0 } },
   { 1, {  1,  2, 1, 0 } },
   { 0, { 13, 14, 0, 0 } },
   { 0, { 14, 15, 0, 0 } },
   { 0, { 15, 16, 0, 0 } },
   { 0, { 16, 17, 0, 0 } },
   { 0, { 17, 18, 0, 0 } },
   { 0, { 18, 19, 0, 0 } },
   { 0, { 19, 20, 0, 0 } },
   { 0, { 20, 21, 0, 0 } },
   { 0, { 21, 22, 0, 0 } },
   { 0, { 22, 23, 0, 0 } },
   { 0, { 23, 24, 0, 0 } },
   { 0, { 24, 25, 0, 0 } },
   { 0, { 25, 26, 0, 0 } },
   { 1, {  0,  0, 2, 1 } },
   { 1, {  0,  1, 2, 1 } },
   { 1, {  1,  2, 0, 0 } },
   { 1, {  0,  1, 1, 2 } },
   { 1, {  2,  1, 1, 0 } },
   { 1, {  0,  0, 2, 0 } },
   { 1, {  0,  2, 1, 0 } },
   { 1, {  0,  1, 2, 0 } },
   { 1, {  0,  2, 0, 0 } },
   { 1, {  0,  1, 0, 2 } },
   { 1, {  2,  0, 1, 0 } },
   { 1, {  1,  2, 1, 1 } },
   { 1, {  0,  2, 1, 1 } },
   { 1, {  1,  1, 2, 0 } },
   { 1, {  1,  1, 2, 1 } },
   { 0, { 11, 12, 0, 0 } },
   { 0, { 12, 13, 0, 0 } },
   { 0, { 13, 14, 0, 0 } },
   { 0, { 14, 15, 0, 0 } },
   { 0, { 15, 16, 0, 0 } },
   { 0, { 16, 17, 0, 0 } },
   { 0, { 17, 18, 0, 0 } },
   { 0, { 18, 19, 0, 0 } },
   { 0, { 19, 20, 0, 0 } },
   { 0, { 20, 21, 0, 0 } },
   { 0, { 21, 22, 0, 0 } },
   { 1, {  1,  2,  0,  1 } },
   { 1, {  1,  0,  2,  0 } },
   { 1, {  1,  0,  2,  1 } },
   { 1, {  0,  2,  0,  1 } },
   { 1, {  2,  1,  1,  1 } },
   { 1, {  1,  1,  1,  2 } },
   { 1, {  2,  1,  0,  1 } },
   { 1, {  1,  0,  1,  2 } },
   { 1, {  0,  0,  2,  2 } },
   { 1, {  0,  1,  2,  2 } },
   { 1, {  2,  2,  1,  0 } },
   { 1, {  1,  2,  2,  0 } },
   { 1, {  1,  0,  0,  2 } },
   { 1, {  2,  0,  0,  1 } },
   { 1, {  0,  2,  2,  1 } },
   { 0, {  7,  8,  0,  0 } },
   { 0, {  8,  9,  0,  0 } },
   { 0, {  9, 10,  0,  0 } },
   { 0, { 10, 11,  0,  0 } },
   { 0, { 11, 12,  0,  0 } },
   { 0, { 12, 13,  0,  0 } },
   { 0, { 13, 14,  0,  0 } },
   { 1, {  2,  2,  0,  0 } },
   { 1, {  1,  2,  2,  1 } },
   { 1, {  1,  1,  0,  2 } },
   { 1, {  2,  0,  1,  1 } },
   { 1, {  1,  1,  2,  2 } },
   { 1, {  2,  2,  1,  1 } },
   { 1, {  0,  2,  2,  0 } },
   { 1, {  0,  2,  1,  2 } },
   { 0, {  6,  7,  0,  0 } },
   { 0, {  7,  8,  0,  0 } },
   { 0, {  8,  9,  0,  0 } },
   { 0, {  9, 10,  0,  0 } },
   { 0, { 10, 11,  0,  0 } },
   { 0, { 11, 12,  0,  0 } },
   { 1, {  1,  0,  2,  2 } },
   { 1, {  2,  2,  0,  1 } },
   { 1, {  2,  1,  2,  0 } },
   { 1, {  2,  2,  2,  0 } },
   { 1, {  0,  2,  2,  2 } },
   { 1, {  2,  2,  2,  1 } },
   { 1, {  2,  1,  2,  1 } },
   { 1, {  1,  2,  1,  2 } },
   { 1, {  1,  2,  2,  2 } },
   { 0, {  3,  4,  0,  0 } },
   { 0, {  4,  5,  0,  0 } },
   { 0, {  5,  6,  0,  0 } },
   { 1, {  0,  2,  0,  2 } },
   { 1, {  2,  0,  2,  0 } },
   { 1, {  1,  2,  0,  2 } },
   { 0, {  3,  4,  0,  0 } },
   { 0, {  4,  5,  0,  0 } },
   { 0, {  5,  6,  0,  0 } },
   { 1, {  2,  0,  2,  1 } },
   { 1, {  2,  1,  1,  2 } },
   { 1, {  2,  1,  0,  2 } },
   { 0, {  3,  4,  0,  0 } },
   { 0, {  4,  5,  0,  0 } },
   { 0, {  5,  6,  0,  0 } },
   { 1, {  2,  2,  2,  2 } },
   { 1, {  2,  2,  1,  2 } },
   { 1, {  2,  1,  2,  2 } },
   { 1, {  2,  0,  1,  2 } },
   { 1, {  2,  0,  0,  2 } },
   { 0, {  1,  2,  0,  0 } },
   { 1, {  2,  2,  0,  2 } },
   { 1, {  2,  0,  2,  2 } } 
};



/* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -
   Codebooks using 2-step lookup method
*/



/* 4.A, Table 4.A.2 */
static const MLO_HcbQuad_Step1   MLO_HcbQuad_coodbook_1_1 [32] =
{
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  1, 0 },
   {  2, 0 },
   {  3, 0 },
   {  4, 0 },
   {  5, 0 },
   {  6, 0 },
   {  7, 0 },
   {  8, 0 },

   /* 7 bit codewords */
   {  9, 2 },
   { 13, 2 },
   { 17, 2 },
   { 21, 2 },
   { 25, 2 },
   { 29, 2 },

   /* 9 bit codewords */
   { 33, 4 },

   /* 9/10/11 bit codewords */
   { 49, 6 }
};

static const MLO_HcbQuad_Step2   MLO_HcbQuad_coodbook_1_2 [] =
{
   /* 1 bit codeword */
   {  1,  0,  0,  0,  0 },

   /* 5 bit codewords */
   {  5,  1,  0,  0,  0 },
   {  5, -1,  0,  0,  0 },
   {  5,  0,  0,  0, -1 },
   {  5,  0,  1,  0,  0 },
   {  5,  0,  0,  0,  1 },
   {  5,  0,  0, -1,  0 },
   {  5,  0,  0,  1,  0 },
   {  5,  0, -1,  0,  0 },

   /* 7 bit codewords */
   /* first 5 bits: 11000 */
   {  7,  1, -1,  0,  0 },
   {  7, -1,  1,  0,  0 },
   {  7,  0,  0, -1,  1 },
   {  7,  0,  1, -1,  0 },
   /* first 5 bits: 11001 */
   {  7,  0, -1,  1,  0 },
   {  7,  0,  0,  1, -1 },
   {  7,  1,  1,  0,  0 },
   {  7,  0,  0, -1, -1 },
   /* first 5 bits: 11010 */
   {  7, -1, -1,  0,  0 },
   {  7,  0, -1, -1,  0 },
   {  7,  1,  0, -1,  0 },
   {  7,  0,  1,  0, -1 },
   /* first 5 bits: 11011 */
   {  7, -1,  0,  1,  0 },
   {  7,  0,  0,  1,  1 },
   {  7,  1,  0,  1,  0 },
   {  7,  0, -1,  0,  1 },
   /* first 5 bits: 11100 */
   {  7,  0,  1,  1,  0 },
   {  7,  0,  1,  0,  1 },
   {  7, -1,  0, -1,  0 },
   {  7,  1,  0,  0,  1 },
   /* first 5 bits: 11101 */
   {  7, -1,  0,  0, -1 },
   {  7,  1,  0,  0, -1 },
   {  7, -1,  0,  0,  1 },
   {  7,  0, -1,  0, -1 },

   /* 9 bit codeword */
   /* first 5 bits: 11110 */
   {  9,  1,  1, -1,  0 },
   {  9, -1,  1, -1,  0 },
   {  9,  1, -1,  1,  0 },
   {  9,  0,  1,  1, -1 },
   {  9,  0,  1, -1,  1 },
   {  9,  0, -1,  1,  1 },
   {  9,  0, -1,  1, -1 },
   {  9,  1, -1, -1,  0 },
   {  9,  1,  0, -1,  1 },
   {  9,  0,  1, -1, -1 },
   {  9, -1,  1,  1,  0 },
   {  9, -1,  0,  1, -1 },
   {  9, -1, -1,  1,  0 },
   {  9,  0, -1, -1,  1 },
   {  9,  1, -1,  0,  1 },
   {  9,  1, -1,  0, -1 },

   /* 9/10/11 bit codewords */
   /* first 5 bits: 11111 */
   /* 9 bit: reading 11 bits -> 2 too much so 4 entries for each codeword */
   {  9, -1,  1,  0, -1 }, {  9, -1,  1,  0, -1 }, {  9, -1,  1,  0, -1 }, {  9, -1,  1,  0, -1 },
   {  9, -1, -1, -1,  0 }, {  9, -1, -1, -1,  0 }, {  9, -1, -1, -1,  0 }, {  9, -1, -1, -1,  0 },
   {  9,  0, -1, -1, -1 }, {  9,  0, -1, -1, -1 }, {  9,  0, -1, -1, -1 }, {  9,  0, -1, -1, -1 },
   {  9,  0,  1,  1,  1 }, {  9,  0,  1,  1,  1 }, {  9,  0,  1,  1,  1 }, {  9,  0,  1,  1,  1 },
   {  9,  1,  0,  1, -1 }, {  9,  1,  0,  1, -1 }, {  9,  1,  0,  1, -1 }, {  9,  1,  0,  1, -1 },
   {  9,  1,  1,  0,  1 }, {  9,  1,  1,  0,  1 }, {  9,  1,  1,  0,  1 }, {  9,  1,  1,  0,  1 },
   {  9, -1,  1,  0,  1 }, {  9, -1,  1,  0,  1 }, {  9, -1,  1,  0,  1 }, {  9, -1,  1,  0,  1 },
   {  9,  1,  1,  1,  0 }, {  9,  1,  1,  1,  0 }, {  9,  1,  1,  1,  0 }, {  9,  1,  1,  1,  0 },
   /* 10 bit: reading 11 bits -> 1 too much so 2 entries for each codeword */
   { 10, -1, -1,  0,  1 }, { 10, -1, -1,  0,  1 },
   { 10, -1,  0, -1, -1 }, { 10, -1,  0, -1, -1 },
   { 10,  1,  1,  0, -1 }, { 10,  1,  1,  0, -1 },
   { 10,  1,  0, -1, -1 }, { 10,  1,  0, -1, -1 },
   { 10, -1,  0, -1,  1 }, { 10, -1,  0, -1,  1 },
   { 10, -1, -1,  0, -1 }, { 10, -1, -1,  0, -1 },
   { 10, -1,  0,  1,  1 }, { 10, -1,  0,  1,  1 },
   { 10,  1,  0,  1,  1 }, { 10,  1,  0,  1,  1 },
   /* 11 bit */
   { 11,  1, -1,  1, -1 },
   { 11, -1,  1, -1,  1 },
   { 11, -1,  1,  1, -1 },
   { 11,  1, -1, -1,  1 },
   { 11,  1,  1,  1,  1 },
   { 11, -1, -1,  1,  1 },
   { 11,  1,  1, -1, -1 },
   { 11, -1, -1,  1, -1 },
   { 11, -1, -1, -1, -1 },
   { 11,  1,  1, -1,  1 },
   { 11,  1, -1,  1,  1 },
   { 11, -1,  1,  1,  1 },
   { 11, -1,  1, -1, -1 },
   { 11, -1, -1, -1,  1 },
   { 11,  1, -1, -1, -1 },
   { 11,  1,  1,  1, -1 }
};



/* 4.A, Table 4.A.3 */
static const MLO_HcbQuad_Step1   MLO_HcbQuad_coodbook_2_1 [32] =
{
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  0, 0 },
   {  1, 0 },
   {  1, 0 },
   {  2, 0 },
   {  3, 0 },
   {  4, 0 },
   {  5, 0 },
   {  6, 0 },
   {  7, 0 },
   {  8, 0 },

   /* 6 bit codewords */
   {  9, 1 },
   { 11, 1 },
   { 13, 1 },
   { 15, 1 },
   { 17, 1 },
   { 19, 1 },
   { 21, 1 },
   { 23, 1 },
   { 25, 1 },
   { 27, 1 },
   { 29, 1 },
   { 31, 1 },

   /* 7 bit codewords */
   { 33, 2 },
   { 37, 2 },
   { 41, 2 },

   /* 7/8 bit codewords */
   { 45, 3 },

   /* 8 bit codewords */
   { 53, 3 },
   { 61, 3 },

   /* 8/9 bit codewords */
   { 69, 4 }
};

static const MLO_HcbQuad_Step2   MLO_HcbQuad_coodbook_2_2 [] =
{
   /* 3 bit codeword */
   { 3,  0,  0,  0,  0 },

   /* 4 bit codeword */
   { 4,  1,  0,  0,  0 },

   /* 5 bit codewords */
   { 5, -1,  0,  0,  0 },
   { 5,  0,  0,  0,  1 },
   { 5,  0,  0, -1,  0 },
   { 5,  0,  0,  0, -1 },
   { 5,  0, -1,  0,  0 },
   { 5,  0,  0,  1,  0 },
   { 5,  0,  1,  0,  0 },

   /* 6 bit codewords */
   { 6,  0, -1,  1,  0 },
   { 6, -1,  1,  0,  0 },
   { 6,  0,  1, -1,  0 },
   { 6,  0,  0,  1, -1 },
   { 6,  0,  1,  0, -1 },
   { 6,  0,  0, -1,  1 },
   { 6, -1,  0,  0, -1 },
   { 6,  1, -1,  0,  0 },
   { 6,  1,  0, -1,  0 },
   { 6, -1, -1,  0,  0 },
   { 6,  0,  0, -1, -1 },
   { 6,  1,  0,  1,  0 },
   { 6,  1,  0,  0,  1 },
   { 6,  0, -1,  0,  1 },
   { 6, -1,  0,  1,  0 },
   { 6,  0,  1,  0,  1 },
   { 6,  0, -1, -1,  0 },
   { 6, -1,  0,  0,  1 },
   { 6,  0, -1,  0, -1 },
   { 6, -1,  0, -1,  0 },
   { 6,  1,  1,  0,  0 },
   { 6,  0,  1,  1,  0 },
   { 6,  0,  0,  1,  1 },
   { 6,  1,  0,  0, -1 },

   /* 7 bit codewords */
   { 7,  0,  1, -1,  1 },
   { 7,  1,  0, -1,  1 },
   { 7, -1,  1, -1,  0 },
   { 7,  0, -1,  1, -1 },
   { 7,  1, -1,  1,  0 },
   { 7,  1,  1,  0, -1 },
   { 7,  1,  0,  1,  1 },
   { 7, -1,  1,  1,  0 },
   { 7,  0, -1, -1,  1 },
   { 7,  1,  1,  1,  0 },
   { 7, -1,  0,  1, -1 },
   { 7, -1, -1, -1,  0 },

   /* 7/8 bit codewords */
   { 7, -1,  0, -1,  1 }, { 7, -1,  0, -1,  1 },
   { 7,  1, -1, -1,  0 }, { 7,  1, -1, -1,  0 },
   { 7,  1,  1, -1,  0 }, { 7,  1,  1, -1,  0 },
   { 8,  1, -1,  0,  1 },
   { 8, -1,  1,  0, -1 },

   /* 8 bit codewords */
   { 8, -1, -1,  1,  0 },
   { 8, -1,  0,  1,  1 },
   { 8, -1, -1,  0,  1 },
   { 8, -1, -1,  0, -1 },
   { 8,  0, -1, -1, -1 },
   { 8,  1,  0,  1, -1 },
   { 8,  1,  0, -1, -1 },
   { 8,  0,  1, -1, -1 },
   { 8,  0,  1,  1,  1 },
   { 8, -1,  1,  0,  1 },
   { 8, -1,  0, -1, -1 },
   { 8,  0,  1,  1, -1 },
   { 8,  1, -1,  0, -1 },
   { 8,  0, -1,  1,  1 },
   { 8,  1,  1,  0,  1 },
   { 8,  1, -1,  1, -1 },

   /* 8/9 bit codewords */
   { 8, -1,  1, -1,  1 }, { 8, -1,  1, -1,  1 },
   { 9,  1, -1, -1,  1 },
   { 9, -1, -1, -1, -1 },
   { 9, -1,  1,  1, -1 },
   { 9, -1,  1,  1,  1 },
   { 9,  1,  1,  1,  1 },
   { 9, -1, -1,  1, -1 },
   { 9,  1, -1,  1,  1 },
   { 9, -1,  1, -1, -1 },
   { 9, -1, -1,  1,  1 },
   { 9,  1,  1, -1, -1 },
   { 9,  1, -1, -1, -1 },
   { 9, -1, -1, -1,  1 },
   { 9,  1,  1, -1,  1 },
   { 9,  1,  1,  1, -1 }
};



/* 4.A, Table 4.A.5 */
static const MLO_HcbQuad_Step1   MLO_HcbQuad_coodbook_4_1 [32] =
{
   /* 4 bit codewords */
   {  0, 0 },
   {  0, 0 },
   {  1, 0 },
   {  1, 0 },
   {  2, 0 },
   {  2, 0 },
   {  3, 0 },
   {  3, 0 },
   {  4, 0 },
   {  4, 0 },
   {  5, 0 },
   {  5, 0 },
   {  6, 0 },
   {  6, 0 },
   {  7, 0 },
   {  7, 0 },
   {  8, 0 },
   {  8, 0 },
   {  9, 0 },
   {  9, 0 },

   /* 5 bit codewords */
   { 10, 0 },
   { 11, 0 },
   { 12, 0 },
   { 13, 0 },
   { 14, 0 },
   { 15, 0 },

   /* 7 bit codewords */
   { 16, 2 },
   { 20, 2 },

   /* 7/8 bit codewords */
   { 24, 3 },

   /* 8 bit codewords */
   { 32, 3 },

   /* 8/9 bit codewords */
   { 40, 4 },

   /* 9/10/11/12 bit codewords */
   { 56, 7 }
};

static const MLO_HcbQuad_Step2   MLO_HcbQuad_coodbook_4_2 [] =
{
   /* 4 bit codewords */
   {  4,  1,  1,  1,  1 },
   {  4,  0,  1,  1,  1 },
   {  4,  1,  1,  0,  1 },
   {  4,  1,  1,  1,  0 },
   {  4,  1,  0,  1,  1 },
   {  4,  1,  0,  0,  0 },
   {  4,  1,  1,  0,  0 },
   {  4,  0,  0,  0,  0 },
   {  4,  0,  0,  1,  1 },
   {  4,  1,  0,  1,  0 },

   /* 5 bit codewords */
   {  5,  1,  0,  0,  1 },
   {  5,  0,  1,  1,  0 },
   {  5,  0,  0,  0,  1 },
   {  5,  0,  1,  0,  1 },
   {  5,  0,  0,  1,  0 },
   {  5,  0,  1,  0,  0 },

   /* 7 bit codewords */
   /* first 5 bits: 11010 */
   {  7,  2,  1,  1,  1 },
   {  7,  1,  1,  2,  1 },
   {  7,  1,  2,  1,  1 },
   {  7,  1,  1,  1,  2 },
   /* first 5 bits: 11011 */
   {  7,  2,  1,  1,  0 },
   {  7,  2,  1,  0,  1 },
   {  7,  1,  2,  1,  0 },
   {  7,  2,  0,  1,  1 },

   /* 7/8 bit codewords */
   /* first 5 bits: 11100 */
   {  7,  0,  1,  2,  1 }, { 7,  0,  1,  2,  1 },
   {  8,  0,  1,  1,  2 },
   {  8,  1,  1,  2,  0 },
   {  8,  0,  2,  1,  1 },
   {  8,  1,  0,  1,  2 },
   {  8,  1,  2,  0,  1 },
   {  8,  1,  1,  0,  2 },

   /* 8 bit codewords */
   {  8,  1,  0,  2,  1 },
   {  8,  2,  1,  0,  0 },
   {  8,  2,  0,  1,  0 },
   {  8,  1,  2,  0,  0 },
   {  8,  2,  0,  0,  1 },
   {  8,  0,  1,  0,  2 },
   {  8,  0,  2,  1,  0 },
   {  8,  0,  0,  1,  2 },

   /* 8/9 bit codewords */
   {  8,  0,  1,  2,  0 }, { 8,  0,  1,  2,  0 },
   {  8,  0,  2,  0,  1 }, { 8,  0,  2,  0,  1 },
   {  8,  1,  0,  0,  2 }, { 8,  1,  0,  0,  2 },
   {  8,  0,  0,  2,  1 }, { 8,  0,  0,  2,  1 },
   {  8,  1,  0,  2,  0 }, { 8,  1,  0,  2,  0 },
   {  8,  2,  0,  0,  0 }, { 8,  2,  0,  0,  0 },
   {  8,  0,  0,  0,  2 }, { 8,  0,  0,  0,  2 },
   {  9,  0,  2,  0,  0 },
   {  9,  0,  0,  2,  0 },

   /* 9/10/11 bit codewords */
   /* 9 bit codewords repeated 2^3 = 8 times */
   {  9,  1,  2,  2,  1 }, {  9,  1,  2,  2,  1 }, {  9,  1,  2,  2,  1 }, {  9,  1,  2,  2,  1 },
   {  9,  1,  2,  2,  1 }, {  9,  1,  2,  2,  1 }, {  9,  1,  2,  2,  1 }, {  9,  1,  2,  2,  1 },
   {  9,  2,  2,  1,  1 }, {  9,  2,  2,  1,  1 }, {  9,  2,  2,  1,  1 }, {  9,  2,  2,  1,  1 },
   {  9,  2,  2,  1,  1 }, {  9,  2,  2,  1,  1 }, {  9,  2,  2,  1,  1 }, {  9,  2,  2,  1,  1 },
   {  9,  2,  1,  2,  1 }, {  9,  2,  1,  2,  1 }, {  9,  2,  1,  2,  1 }, {  9,  2,  1,  2,  1 },
   {  9,  2,  1,  2,  1 }, {  9,  2,  1,  2,  1 }, {  9,  2,  1,  2,  1 }, {  9,  2,  1,  2,  1 },
   {  9,  1,  1,  2,  2 }, {  9,  1,  1,  2,  2 }, {  9,  1,  1,  2,  2 }, {  9,  1,  1,  2,  2 },
   {  9,  1,  1,  2,  2 }, {  9,  1,  1,  2,  2 }, {  9,  1,  1,  2,  2 }, {  9,  1,  1,  2,  2 },
   {  9,  1,  2,  1,  2 }, {  9,  1,  2,  1,  2 }, {  9,  1,  2,  1,  2 }, {  9,  1,  2,  1,  2 },
   {  9,  1,  2,  1,  2 }, {  9,  1,  2,  1,  2 }, {  9,  1,  2,  1,  2 }, {  9,  1,  2,  1,  2 },
   {  9,  2,  1,  1,  2 }, {  9,  2,  1,  1,  2 }, {  9,  2,  1,  1,  2 }, {  9,  2,  1,  1,  2 },
   {  9,  2,  1,  1,  2 }, {  9,  2,  1,  1,  2 }, {  9,  2,  1,  1,  2 }, {  9,  2,  1,  1,  2 },
   /* 10 bit codewords repeated 2^2 = 4 times */
   { 10,  1,  2,  2,  0 }, { 10,  1,  2,  2,  0 }, { 10,  1,  2,  2,  0 }, { 10,  1,  2,  2,  0 },
   { 10,  2,  2,  1,  0 }, { 10,  2,  2,  1,  0 }, { 10,  2,  2,  1,  0 }, { 10,  2,  2,  1,  0 },
   { 10,  2,  1,  2,  0 }, { 10,  2,  1,  2,  0 }, { 10,  2,  1,  2,  0 }, { 10,  2,  1,  2,  0 },
   { 10,  0,  2,  2,  1 }, { 10,  0,  2,  2,  1 }, { 10,  0,  2,  2,  1 }, { 10,  0,  2,  2,  1 },
   { 10,  0,  1,  2,  2 }, { 10,  0,  1,  2,  2 }, { 10,  0,  1,  2,  2 }, { 10,  0,  1,  2,  2 },
   { 10,  2,  2,  0,  1 }, { 10,  2,  2,  0,  1 }, { 10,  2,  2,  0,  1 }, { 10,  2,  2,  0,  1 },
   { 10,  0,  2,  1,  2 }, { 10,  0,  2,  1,  2 }, { 10,  0,  2,  1,  2 }, { 10,  0,  2,  1,  2 },
   { 10,  2,  0,  2,  1 }, { 10,  2,  0,  2,  1 }, { 10,  2,  0,  2,  1 }, { 10,  2,  0,  2,  1 },
   { 10,  1,  0,  2,  2 }, { 10,  1,  0,  2,  2 }, { 10,  1,  0,  2,  2 }, { 10,  1,  0,  2,  2 },
   { 10,  2,  2,  2,  1 }, { 10,  2,  2,  2,  1 }, { 10,  2,  2,  2,  1 }, { 10,  2,  2,  2,  1 },
   { 10,  1,  2,  0,  2 }, { 10,  1,  2,  0,  2 }, { 10,  1,  2,  0,  2 }, { 10,  1,  2,  0,  2 },
   { 10,  2,  0,  1,  2 }, { 10,  2,  0,  1,  2 }, { 10,  2,  0,  1,  2 }, { 10,  2,  0,  1,  2 },
   { 10,  2,  1,  0,  2 }, { 10,  2,  1,  0,  2 }, { 10,  2,  1,  0,  2 }, { 10,  2,  1,  0,  2 },
   { 10,  1,  2,  2,  2 }, { 10,  1,  2,  2,  2 }, { 10,  1,  2,  2,  2 }, { 10,  1,  2,  2,  2 },
   /* 11 bit codewords repeated 2^1 = 2 times */
   { 11,  2,  1,  2,  2 }, { 11,  2,  1,  2,  2 },
   { 11,  2,  2,  1,  2 }, { 11,  2,  2,  1,  2 },
   { 11,  0,  2,  2,  0 }, { 11,  0,  2,  2,  0 },
   { 11,  2,  2,  0,  0 }, { 11,  2,  2,  0,  0 },
   { 11,  0,  0,  2,  2 }, { 11,  0,  0,  2,  2 },
   { 11,  2,  0,  2,  0 }, { 11,  2,  0,  2,  0 },
   { 11,  0,  2,  0,  2 }, { 11,  0,  2,  0,  2 },
   { 11,  2,  0,  0,  2 }, { 11,  2,  0,  0,  2 },
   { 11,  2,  2,  2,  2 }, { 11,  2,  2,  2,  2 },
   { 11,  0,  2,  2,  2 }, { 11,  0,  2,  2,  2 },
   { 11,  2,  2,  2,  0 }, { 11,  2,  2,  2,  0 },
   /* 12 bit codewords */
   { 12,  2,  2,  0,  2 },
   { 12,  2,  0,  2,  2 },
};



/* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -
   Codebook list
*/



static const MLO_HcbQuad_Index   MLO_HcbQuad_cookbook_index [MLO_HCB_FIRST_PAIR_HCB] =
{
   { 0, 0, 0 },
   { 0, MLO_HcbQuad_coodbook_1_1, MLO_HcbQuad_coodbook_1_2 },
   { 0, MLO_HcbQuad_coodbook_2_1, MLO_HcbQuad_coodbook_2_2 },
   { MLO_HcbQuad_coodbook_3, 0, 0 },
   { 0, MLO_HcbQuad_coodbook_4_1, MLO_HcbQuad_coodbook_4_2 }
};



/*----------------------------------------------------------------------
|       Functions
+---------------------------------------------------------------------*/



/*
==============================================================================
Name: MLO_HcbQuad_decode_binary
Description:
   Decodes 4 Huffman-compressed values from the stream, using the binary
   search tree method.
Input parameters:
	- data_ptr: Points on the first value location in the spectral data array.
	- hcb: Huffman CodeBook to use. Only Codebook 3 is allowed.
Input/output parameters:
	- bit_ptr: Input bitstream to decode.
==============================================================================
*/

void  
MLO_HcbQuad_decode_binary (MLO_Int16 data_ptr [4], MLO_BitStream *bit_ptr, MLO_Hcb hcb)
{
   unsigned int offset = 0;
   const MLO_HcbQuad_Binary * table_ptr;

   MLO_ASSERT(data_ptr != NULL);
   MLO_ASSERT(bit_ptr != NULL);
   MLO_ASSERT(hcb > MLO_HCB_ZERO_HCB);
   MLO_ASSERT(hcb < MLO_HCB_FIRST_PAIR_HCB);

   table_ptr = MLO_HcbQuad_cookbook_index [hcb].bin_ptr;
   MLO_ASSERT(table_ptr != NULL);

   while (! table_ptr [offset].leaf_flag)
   {
      unsigned int bit = MLO_BitStream_ReadBit (bit_ptr);
      offset += table_ptr [offset].data [bit];
   }

   data_ptr [0] = table_ptr [offset].data [0];
   data_ptr [1] = table_ptr [offset].data [1];
   data_ptr [2] = table_ptr [offset].data [2];
   data_ptr [3] = table_ptr [offset].data [3];
}



/*
==============================================================================
Name: MMLO_HcbQuad_decode_2steps
Description:
   Decodes 4 Huffman-compressed values from the stream, using the 2-step
   method.
Input parameters:
	- data_ptr: Points on the first value location in the spectral data array.
	- hcb: Huffman CodeBook to use. Only Codebooks 1, 2 and 4 are allowed.
Input/output parameters:
	- bit_ptr: Input bitstream to decode.
==============================================================================
*/

void  
MLO_HcbQuad_decode_2steps (MLO_Int16 data_ptr [4], MLO_BitStream *bit_ptr, MLO_Hcb hcb)
{
   int            offset = 0;
   int            nbr_bits = 5;
   int            cw;
   int            extra_bits;
   const MLO_HcbQuad_Step1 *  table_1_ptr;
   const MLO_HcbQuad_Step2 *  table_2_ptr;

   MLO_ASSERT(data_ptr != NULL);
   MLO_ASSERT(bit_ptr != NULL);
   MLO_ASSERT(hcb > MLO_HCB_ZERO_HCB);
   MLO_ASSERT(hcb < MLO_HCB_FIRST_PAIR_HCB);

   table_1_ptr = MLO_HcbQuad_cookbook_index [hcb].step_1_ptr;
   table_2_ptr = MLO_HcbQuad_cookbook_index [hcb].step_2_ptr;
   MLO_ASSERT (table_1_ptr != NULL);
   MLO_ASSERT (table_2_ptr != NULL);

   cw = MLO_BitStream_PeekBits (bit_ptr, nbr_bits);

   offset     = table_1_ptr [cw].offset;
   extra_bits = table_1_ptr [cw].extra_bits;

   if (extra_bits == 0)
   {
      MLO_BitStream_SkipBits (bit_ptr, table_2_ptr [offset].bits);
   }
   else
   {
      MLO_BitStream_SkipBits (bit_ptr, nbr_bits);
      offset += MLO_BitStream_PeekBits (bit_ptr, extra_bits);
      MLO_BitStream_SkipBits (bit_ptr, table_2_ptr [offset].bits - nbr_bits);
   }

   data_ptr [0] = table_2_ptr [offset].x;
   data_ptr [1] = table_2_ptr [offset].y;
   data_ptr [2] = table_2_ptr [offset].z;
   data_ptr [3] = table_2_ptr [offset].w;
}



